#!/bin/bash

# Nom du fichier d'entrée généré par le script précédent
INPUT_FILE="project_context.txt"

# Début du séparateur utilisé dans le fichier de contexte (doit correspondre)
SEPARATOR="----------------------------------------------------"

# --- Début du script ---

echo "Restauration du code à partir de $INPUT_FILE..."

# Vérifier si le fichier d'entrée existe
if [ ! -f "$INPUT_FILE" ]; then
    echo "Erreur: Le fichier d'entrée '$INPUT_FILE' n'existe pas."
    exit 1
fi

# Variables de contrôle
current_file=""
content_buffer=""
in_content=0 # 0=recherche de séparateur/chemin, 1=lecture du contenu

# Lire le fichier ligne par ligne
# IFS= lit sans interpréter les sauts de ligne ou les espaces spéciaux
# -r empêche l'interprétation des barres obliques inverses
while IFS= read -r line; do
    # Supprimer les espaces de début et de fin pour la comparaison
    trimmed_line=$(echo "$line" | sed 's/^[[:space:]]*//;s/[[:space:]]*$//')

    if [ "$trimmed_line" = "$SEPARATOR" ]; then
        # On a trouvé un séparateur : soit on commence un nouveau fichier, soit on termine l'ancien.
        if [ "$in_content" -eq 1 ] && [ -n "$current_file" ]; then
            # --- Écriture du fichier précédent ---

            # Créer les répertoires nécessaires
            # dirname extrait le chemin du répertoire du nom de fichier
            DIRNAME=$(dirname "$current_file")
            if [ ! -d "$DIRNAME" ]; then
                mkdir -p "$DIRNAME"
                echo "-> Création du répertoire: $DIRNAME"
            fi

            # Écrire le contenu dans le fichier
            echo "$content_buffer" > "$current_file"
            echo "✅ Fichier restauré: $current_file (${#content_buffer} octets)"

            # Réinitialiser pour le fichier suivant
            current_file=""
            content_buffer=""
            in_content=0
        fi

        # La ligne suivante devrait être le chemin d'accès
        continue

    elif [ "$in_content" -eq 0 ] && [ -z "$current_file" ] && [ -n "$trimmed_line" ]; then
        # On a trouvé le chemin d'accès après le séparateur (et la ligne vide intermédiaire)

        # On suppose que la première ligne non vide après le séparateur est le chemin.
        # Le script original écrit : SEPARATOR, CHEMIN, LIGNE_VIDE, CONTENU.
        # Donc, si on est en "in_content=0" et on vient de lire SEPARATOR (juste avant de lire cette ligne),
        # cette ligne est le chemin.
        current_file="$trimmed_line"
        in_content=0 # Reste à 0 pour ignorer la ligne vide suivante

    elif [ -n "$current_file" ]; then
        # Si on a un chemin, on commence à accumuler le contenu

        # Si la première ligne après le chemin et la ligne vide est rencontrée,
        # on passe en mode lecture de contenu.
        # La logique de séparation dans le script initial rend la détection du début du contenu un peu tricky :
        # SEPARATOR -> CHEMIN -> LIGNE_VIDE -> CONTENU...
        
        # Le moyen le plus simple est de commencer à accumuler tout le reste.
        
        # Le script original ajoute une ligne vide après le chemin.
        # Il faut donc ajouter la ligne lue au buffer, en tenant compte des sauts de ligne.
        
        if [ "$in_content" -eq 0 ]; then
             # Premier appel pour le contenu (après le chemin et la ligne vide),
             # on passe en mode lecture de contenu.
             in_content=1
        fi
        
        # Le `cat` du script original n'ajoute pas de saut de ligne final s'il n'y en a pas.
        # Pour une fidélité maximale : accumuler la ligne telle quelle, et ajouter un saut de ligne.
        if [ -n "$content_buffer" ]; then
            content_buffer+=$'\n'
        fi
        
        # Le `read -r line` supprime le caractère de saut de ligne final.
        # Nous devons le rajouter manuellement pour la reconstruction.
        content_buffer+="$line"

    fi
done < "$INPUT_FILE"

# --- Gestion du dernier fichier ---
# Après la fin de la boucle, si on a encore du contenu en mémoire
if [ "$in_content" -eq 1 ] && [ -n "$current_file" ]; then
    DIRNAME=$(dirname "$current_file")
    if [ ! -d "$DIRNAME" ]; then
        mkdir -p "$DIRNAME"
        echo "-> Création du répertoire: $DIRNAME"
    fi

    echo "$content_buffer" > "$current_file"
    echo "✅ Fichier restauré: $current_file (${#content_buffer} octets)"
fi

echo ""
echo "Terminé ! La restauration du code est complète."
