// server/server.js

const express = require('express');
const cors = require('cors');
const helmet = require('helmet');
const dotenv = require('dotenv');
const path = require('path');
const fs = require('fs');

dotenv.config();

const app = express();
const PORT = process.env.PORT || 3000;

// ================================================================
// MIDDLEWARES
// ================================================================
app.set('trust proxy', true);
app.use(helmet({ contentSecurityPolicy: false }));
app.use(cors({ origin: true, credentials: true }));
app.use(express.json({ limit: '10mb' }));
app.use(express.urlencoded({ extended: true, limit: '10mb' }));

// Logger
app.use((req, res, next) => {
    console.log(`[${new Date().toISOString()}] ${req.method} ${req.path}`);
    next();
});

// ================================================================
// GESTION DU MULTILINGUE ET FICHIERS STATIQUES (CORRIGÉ)
// ================================================================
const rootDir = path.join(__dirname, '..');

// 1. Servir les pages d'accueil statiques pour le SEO (fr/, de/, it/, en/)
// Ces routes sont spécifiquement pour les bots qui visitent les URLs de base.
app.use('/fr', express.static(path.join(rootDir, 'public/fr'))); // [cite: 9] (adapté)
app.use('/de', express.static(path.join(rootDir, 'public/de'))); // [cite: 9] (adapté)
app.use('/it', express.static(path.join(rootDir, 'public/it'))); // [cite: 9] (adapté)
app.use('/en', express.static(path.join(rootDir, 'public/en'))); // [cite: 9] (adapté)

// 2. Servir les fichiers statiques communs (CSS, JS, images, etc.)
app.use(express.static(path.join(rootDir, 'public'))); // [cite: 10]

// ================================================================
// CHARGER LES CONFIGS
// ================================================================
let appConfigs = {};
try {
    appConfigs.countries = JSON.parse(fs.readFileSync(path.join(__dirname, '../config/countries.json'), 'utf8'));
    appConfigs.currencies = JSON.parse(fs.readFileSync(path.join(__dirname, '../config/currencies.json'), 'utf8')); // [cite: 13]
    appConfigs.taresRules = JSON.parse(fs.readFileSync(path.join(__dirname, '../config/tares_codes.json'), 'utf8'));
    
    // ... reste du chargement de config ... [cite: 14, 15]
		
    app.appConfigs = appConfigs;
    console.log('✓ Fichiers de configuration chargés'); // [cite: 16]
} catch (error) {
    console.error('⚠ Erreur chargement config:', error.message); // [cite: 17]
}

// ================================================================
// ROUTES DE L'API
// ================================================================
const apiRoutes = require('./routes/api');
app.use('/api', apiRoutes);

// ================================================================
// ROUTE FALLBACK POUR L'APPLICATION ANGULAR (Single Page App)
// Doit être la dernière route. Elle intercepte TOUT ce qui n'est pas
// un fichier statique ou une route API.
// ================================================================
app.get('*', (req, res) => {
    const indexPath = path.join(rootDir, 'public/index.html');
    fs.readFile(indexPath, 'utf8', (err, html) => {
        if (err) {
            return res.status(500).send('Erreur serveur.');
        }
        // Injection de la clé Stripe, comme avant
        const finalHtml = html.replace('{{STRIPE_PUBLIC_KEY}}', process.env.STRIPE_PUBLIC_KEY || '');
        res.send(finalHtml);
    });
});

// ================================================================
// GESTION D'ERREURS (inchangée)
// ================================================================
app.use((err, req, res, next) => {
    // ... votre gestionnaire d'erreurs
});
app.use((req, res) => {
    res.status(404).json({ error: 'Route non trouvée: ' + req.path });
});

// ================================================================
// DÉMARRAGE (inchangé)
// ================================================================
app.listen(PORT, '127.0.0.1', () => {
    console.log(`╔════════════════════════════════════════════════════════════╗`);
    console.log(`║  Backend e-dec démarré sur http://127.0.0.1:${PORT}        ║`);
    console.log(`╚════════════════════════════════════════════════════════════╝`);
});