#!/bin/bash

# Nom du fichier de sortie
OUTPUT_FILE="project_context_recent.txt"

# Dossiers et fichiers à inclure (optionnel - si vide, tous les fichiers modifiés seront inclus)
TARGET_PATHS=("server" "public" "package.json")

# Nombre de jours pour "récemment modifié" (par défaut: 7 jours)
DAYS_BACK=3

# Branche Git à comparer (par défaut: main)
COMPARE_BRANCH="main"

# Fichiers et dossiers à ignorer
EXCLUDE_PATTERNS=(
    "node_modules"
    "logs"
    "backup"
    ".git"
    "*.log"
    "*.lock"
    "*.swp"
    "*.map"
    "*.jpg"
    "*.png"
    "*.gif"
    "*.ico"
    "*.eot"
    "*.ttf"
    "*.woff"
    "*.woff2"
    "*.svg"
    "*.min.js"
    "*.min.css"
    "*.sh"
)

# --- Début du script ---

# Supprimer le fichier de sortie précédent s'il existe
rm -f "$OUTPUT_FILE"

echo "Génération du fichier de contexte pour les fichiers récemment modifiés..."
echo "Période: depuis $DAYS_BACK jours"
echo "Le résultat sera enregistré dans : $OUTPUT_FILE"
echo "" >> "$OUTPUT_FILE"

# Vérifier que nous sommes dans un dépôt Git
if ! git rev-parse --git-dir > /dev/null 2>&1; then
    echo "Erreur: Ce répertoire n'est pas un dépôt Git"
    exit 1
fi

# Fonction pour vérifier si un fichier doit être ignoré
should_exclude() {
    local path="$1"
    for pattern in "${EXCLUDE_PATTERNS[@]}"; do
        if [[ "$path" == *"$pattern"* ]]; then
            return 0 # Oui, exclure
        fi
    done
    return 1 # Non, inclure
}

# Fonction pour vérifier si un fichier est dans les TARGET_PATHS
is_in_target_paths() {
    local file="$1"
    
    # Si TARGET_PATHS est vide, inclure tous les fichiers
    if [ ${#TARGET_PATHS[@]} -eq 0 ]; then
        return 0
    fi
    
    for target in "${TARGET_PATHS[@]}"; do
        if [[ "$file" == "$target"* ]] || [[ "$file" == "$target" ]]; then
            return 0
        fi
    done
    return 1
}

# Récupérer les fichiers modifiés dans les commits récents
echo "Récupération des fichiers modifiés depuis $DAYS_BACK jours..."

# Méthode 1: Fichiers modifiés dans les commits des derniers jours
RECENT_FILES=$(git log --since="$DAYS_BACK.days" --name-only --oneline --pretty=format: | sort -u)

# Méthode 2: Fichiers modifiés dans le working directory (non commités)
UNSTAGED_FILES=$(git status --porcelain | awk '{$1=""; print substr($0,2)}' | sed 's/^ *//')

# Combiner et trier les fichiers
ALL_MODIFIED_FILES=$(echo -e "$RECENT_FILES\n$UNSTAGED_FILES" | grep -v '^$' | sort -u)

# Compteurs
TOTAL_FILES=0
ADDED_FILES=0

# Traiter chaque fichier modifié
while IFS= read -r FILE_PATH; do
    # Vérifier que le fichier existe
    if [ ! -f "$FILE_PATH" ]; then
        continue
    fi
    
    # Vérifier les exclusions et les chemins cibles
    if ! should_exclude "$FILE_PATH" && is_in_target_paths "$FILE_PATH"; then
        echo "Ajout de: $FILE_PATH"
        echo "----------------------------------------------------" >> "$OUTPUT_FILE"
        echo "$FILE_PATH" >> "$OUTPUT_FILE"
        echo "Dernière modification: $(git log -1 --format="%cd" --date=short "$FILE_PATH" 2>/dev/null || stat -c %y "$FILE_PATH" 2>/dev/null || echo "inconnue")" >> "$OUTPUT_FILE"
        echo "" >> "$OUTPUT_FILE"
        
        # Insérer le contenu du fichier
        cat "$FILE_PATH" >> "$OUTPUT_FILE"
        echo "" >> "$OUTPUT_FILE" # Ajouter une ligne vide pour la séparation
        
        ((ADDED_FILES++))
    fi
    ((TOTAL_FILES++))
done <<< "$ALL_MODIFIED_FILES"

# Alternative: Si peu de fichiers trouvés, utiliser git diff avec la branche de comparaison
if [ $ADDED_FILES -eq 0 ]; then
    echo "Aucun fichier récent trouvé avec la méthode des commits, tentative avec git diff..."
    
    DIFF_FILES=$(git diff --name-only "$COMPARE_BRANCH"...HEAD)
    
    while IFS= read -r FILE_PATH; do
        if [ -f "$FILE_PATH" ] && ! should_exclude "$FILE_PATH" && is_in_target_paths "$FILE_PATH"; then
            echo "Ajout de: $FILE_PATH"
            echo "----------------------------------------------------" >> "$OUTPUT_FILE"
            echo "$FILE_PATH" >> "$OUTPUT_FILE"
            echo "(Modifié depuis $COMPARE_BRANCH)" >> "$OUTPUT_FILE"
            echo "" >> "$OUTPUT_FILE"
            
            cat "$FILE_PATH" >> "$OUTPUT_FILE"
            echo "" >> "$OUTPUT_FILE"
            
            ((ADDED_FILES++))
        fi
        ((TOTAL_FILES++))
    done <<< "$DIFF_FILES"
fi

echo ""
echo "Terminé !"
echo "Fichiers analysés: $TOTAL_FILES"
echo "Fichiers ajoutés au contexte: $ADDED_FILES"
echo "Contexte des modifications récentes dans: $OUTPUT_FILE"

# Afficher un résumé des fichiers inclus (CORRIGÉ)
if [ $ADDED_FILES -gt 0 ]; then
    echo ""
    echo "Fichiers récents inclus:"
    # Méthode corrigée pour éviter l'erreur grep
    awk '/^[^ -]/ && !/^$/' "$OUTPUT_FILE" | head -10
    if [ $ADDED_FILES -gt 10 ]; then
        echo "... et $((ADDED_FILES - 10)) autres"
    fi
fi
